/*:
 * @target MZ
 * @plugindesc アクターが特定のスキルを習得している場合、戦闘開始時に確率で自動発動しメッセージを表示します。
 * @author あなたの名前
 * @help
 * アクターが特定のスキルを習得している場合、
 * 戦闘開始時にそのスキルを自動で発動します（演出付き）。
 * さらに、発動確率を設定でき、発動時にはメッセージが表示されます。
 *
 * 【使い方】
 * 1. 自動発動させたいスキルのメモ欄に以下のいずれかを記述してください。
 *    - `<AutoBattleSkill>`: 100%の確率で発動します。
 *    - `<AutoBattleSkill:確率>`: 指定した確率（例: `<AutoBattleSkill:50>` で50%）で発動します。
 *      確率は0〜100の整数で指定します。
 *
 * 例：
 *   - スキル「先制攻撃」のメモ欄に `<AutoBattleSkill>` と記述すると、
 *     「先制攻撃」を覚えているアクターが戦闘開始時に100%の確率で自動でスキルを発動します。
 *   - スキル「防御アップ」のメモ欄に `<AutoBattleSkill:75>` と記述すると、
 *     「防御アップ」を覚えているアクターが戦闘開始時に75%の確率で自動でスキルを発動します。
 *
 * 2. 複数のスキルにタグを設定すれば、それらを覚えている各アクターが
 *    それぞれのスキルを確率に応じて発動し、メッセージを表示します。
 *
 * @command dummy
 * @text 設定項目なし
 * @desc このプラグインに設定項目はありません。
 */

(() => {
    const pluginName = "IndividualAutoSkillOnBattleStartWithProb";
    const autoSkillTag = "AutoBattleSkill";

    const _BattleManager_startBattle = BattleManager.startBattle;
    BattleManager.startBattle = function() {
        _BattleManager_startBattle.call(this);

        let forcedActions = []; // 強制実行するアクションを一時的に保持する配列

        // 戦闘に参加しているパーティメンバー全員をチェック
        $gameParty.battleMembers().forEach(actor => {
            if (actor.isActor()) {
                // アクターが習得しているスキルを全てチェック
                actor.skills().forEach(skill => {
                    if (skill && skill.meta[autoSkillTag]) {
                        // メモ欄のタグから確率を解析
                        let activateRate = 100; // デフォルトは100%
                        const tagValue = skill.meta[autoSkillTag];
                        if (typeof tagValue === 'string') {
                            const rateMatch = tagValue.match(/^(\d+)$/);
                            if (rateMatch) {
                                activateRate = Number(rateMatch[1]);
                            }
                        }

                        // 確率判定[1]
                        if (Math.random() * 100 < activateRate) {
                            // 発動メッセージの表示[2]
                            const actorName = actor.name();
                            const skillName = skill.name;
                            $gameMessage.add(`${actorName}の${skillName}が発動した！`);

                            // 強制アクションを登録
                            actor.forceAction(skill.id, -1); // ターゲットは自動
                            BattleManager.forceAction(actor);
                        }
                    }
                });
            }
        });

        // 強制されたアクションがあれば実行を開始
        // 戦闘開始メッセージの後に、スキル発動メッセージと演出を確実に出すために
        // ここで遅延実行する必要があるが、BattleManager.processForcedAction() は
        // そのまま使うと、戦闘開始時のフェードインと重なる可能性があるため、
        // ターン開始時にまとめて実行されるようにする。
        // ここで直接実行すると、戦闘開始時のシステムメッセージより先に表示される可能性がある。
        // したがって、通常のアクションキューに任せるのが最も安全。
        // 強制アクションは自動的にターン開始時に処理されるため、
        // ここで特別な処理は不要です。
    };
})();
